set timing off verify off echo off tab off pages 999 lines 32767 trimspool on define "&" serverout on

WHENEVER SQLERROR EXIT
WHENEVER OSERROR EXIT

set term off feed off

define iv = '4.2.7'

column cronology_log new_value cronology_log noprint
select 'cronology_server_'||replace('&iv','.','_')||'_patch_'||to_char(sysdate,'YYYYMMDDHH24MISS')||'.log' cronology_log from dual;

set term on

declare
   script_user varchar2(30) := 'CRONOLOGY';
begin
   if user <> script_user then
      raise_application_error(-20000,'Please connect as '||script_user||' to run this script.');
   end if;
end;
/

spool &cronology_log

prompt
prompt Cronology Server &iv Patch
prompt ============================
prompt

var curr_ver   varchar2(1000)

prompt Checking patch prerequisites ...
prompt

declare
   pv             varchar2(10) := '4.2';
begin
   begin
      execute immediate 'begin
                             :v := install_info.server_ver;
                         exception
                            when others then
                              :v := sqlerrm;
                         end;' using in out :curr_ver;
   exception
      when others then
         :curr_ver := sqlerrm;
   end;

-- Assign directly so can also access outside of PL/SQL block
   :curr_ver := :curr_ver;

   if :curr_ver like 'ORA%' then
      raise_application_error(-20000,'Cronology server installation is currently invalid. Patching can only take place on a valid install. Please email support@cronology.co.uk if you require assistance.');
   elsif :curr_ver = '&iv' then
      dbms_output.put_line('Cronology server version '||:curr_ver||' detected - re-applying patch.');
   elsif :curr_ver like pv||'.%' then
      dbms_output.put_line('Cronology server version '||:curr_ver||' detected.');
   else
      raise_application_error(-20000,'Cronology server version '||:curr_ver||' not valid for patching to version &iv Please upgrade to version '||pv||'.0 first.');
   end if;
end;
/

prompt
prompt Stopping Cronology background processes ...
prompt
begin
      begin
      -- This will delete the lgwr pipe
         execute immediate 'begin external.stop_log_writer; end;';
      exception
         when others then
            null;
      end;
      begin
         execute immediate 'begin external.stop_job_poller; end;';
      exception
         when others then
            null;
      end;
      begin
         execute immediate 'begin external.stop_message_server; end;';
      exception
         when others then
            null;
      end;
end;
/
prompt Complete.

declare
   conns boolean := false;
   holder                varchar2(100);
begin
   for i in (select inst_id, username, module, program, osuser, sid, serial#
             from   gv$session
             where  username = 'CRONOLOGY'
             and   (sid, inst_id) not in (select sys_context('USERENV','SID') , sys_context('USERENV','INSTANCE') from dual)) loop
      conns := true;
      dbms_output.put_line(chr(10)||'Instance ID: '||i.inst_id);
      dbms_output.put_line('Username   : '||i.username);
      dbms_output.put_line('Module     : '||i.module);
      dbms_output.put_line('Program    : '||i.program);
      dbms_output.put_line('OS User    : '||i.osuser);
      dbms_output.put_line('SID        : '||i.sid);
      dbms_output.put_line('Serial#    : '||i.serial#);
   end loop;

   if conns then
      dbms_output.put_line(chr(10)||'Cronology database sessions still exist (see above). The Job Poller, Log Writer and Message Server have been stopped so these must be user connections.'||chr(10));
      raise_application_error(-20000,'Please disconnect all Cronology related sessions before patching.');
   end if;
end;
/

prompt
prompt Prerequisite checks complete.
prompt

set feed off pages 0

select 'Patch will take place on '||
upper(substr(host_name,1,decode(instr(host_name,'.'),0,9999,instr(host_name,'.')-1)))||' '||upper(instance_name)
from v$instance;

prompt

----------------------------------------------------------------------------------------------------------------------------------------------
prompt Applying patch ...
prompt
----------------------------------------------------------------------------------------------------------------------------------------------

begin execute immediate 'alter session set deferred_segment_creation=false'; exception when others then null; end;
/

prompt Determining data and index tablespaces ...
prompt
column data_ts new_value data_ts
select tablespace_name data_ts from user_tables where table_name = 'JOB';

column ind_ts new_value ind_ts
select tablespace_name ind_ts from user_indexes where index_name = 'PK_JOB';

set feed on

----------------------------------------------------------------------------------------------------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------

alter table job modify (db_conn_string varchar2(150));
alter table TRACE_HISTORY modify (OSUSER varchar2(128));
alter table TRACE_HISTORY modify (USERNAME varchar2(128));
alter table MESSAGE_HISTORY modify (SENT_BY_USER varchar2(128));
alter table JOB_RUN_HISTORY modify (ACTION_USERNAME varchar2(128));
alter table JOB_RUN_HISTORY modify (RUN_BY varchar2(128));
alter table JOB modify (USERNAME varchar2(128));
alter table CONSOLE_AUTO_TEXT modify (USERNAME varchar2(128));
alter table BG_PROCESS_HISTORY modify (USERNAME varchar2(128));
alter table AUDIT_HISTORY modify (OSUSER varchar2(128));
alter table AUDIT_HISTORY modify (USERNAME varchar2(128));

alter table MESSAGE_HISTORY modify (SENT_BY_MODULE varchar2(64));

begin execute immediate 'drop trigger p_aud_trig'; exception when others then null; end;
/

begin execute immediate 'drop trigger parameters_trigger'; exception when others then null; end;
/


update parameters
set    description = 'An optional label to describe the database environment e.g. Production, Development etc. This will be displayed on the console main screen. Leave as <NONE> if no label required. N.B. This setting will affect all console users on this database.'
||chr(10)||chr(10)||
'Change Effective: Console main page reload required by other users.'
where  name = 'ENVIRONMENT_LABEL'
and    param_id < 0;

update parameters
set    description = 'Select a colour to reflect the environment e.g. blue for development, red for production. This will be displayed on the console main screen banner. Leave as <NONE> if no colour required. N.B. This setting will affect all console users on this database.'
||chr(10)||chr(10)||
'Change Effective: Console main page reload required by other users.',
       value = decode(value,'B','BLUE',
                      'G','GREEN',
                      'O','ORANGE',
                      'R','RED',
                      'Y','YELLOW',
                      value)
where  name = 'ENVIRONMENT_COLOUR'
and    param_id < 0;

-- remove CONSOLE_DATA_UPD_DTTM
delete parameters where param_id = -23;
delete parameters where param_id = -40;

-- reuse the slot
insert into parameters
values (-23, 'FIXED', 'PLATFORM_INFO_EMAIL_FROM', null,
'Specifies whether the e-mail From description shows the host name, instance name, or both.'
||chr(10)||chr(10)||
'Change Effective: Immediate.'
,'CHARACTER', 'HOST', 'CONSOLE','N');

insert into parameters
values (-40, 'FIXED', 'PLATFORM_INFO_MESSAGE', null,
'Specifies whether generic messaging includes the host name, instance name, or both.'
||chr(10)||chr(10)||
'Change Effective: Immediate.'
,'CHARACTER', 'HOST+INST', 'CONSOLE','N');

commit;

create or replace type varchar_t as table of varchar2(4000);
/

create or replace type number_t as table of number;
/

create or replace type src_o FORCE as object (text varchar2(4000), goto_line number);
/

create or replace type src_t FORCE as table of src_o;
/


grant execute on date_t to cronology_readonly;

grant execute on number_t to cronology_readonly;

grant execute on src_o to cronology_readonly;

grant execute on src_t to cronology_readonly;

create or replace view job_email_addresses_v as
select email email_address, listagg(referenced,' & ') within group (order by referenced) referenced
from (
SELECT regexp_substr(j.email, '[^ ]+', 1, rws.column_value) email, 'On Completion' referenced
FROM job j,
TABLE(CAST(MULTISET(SELECT LEVEL
                    FROM   dual
                    CONNECT BY level <= regexp_count(j.email, '[^ ]+')) AS number_t)) rws
where j.email is not null
union
SELECT regexp_substr(j.email_on_failure, '[^ ]+', 1, rws.column_value) email, 'On Failure Override'
FROM job j,
TABLE(CAST(MULTISET(SELECT LEVEL
                    FROM   dual
                    CONNECT BY level <= regexp_count(j.email_on_failure, '[^ ]+')) AS number_t)) rws
where j.email_on_failure is not null)
group by email;

---------------------------------
-- NEW SCHEDULE_SETTINGS START --
---------------------------------

begin execute immediate 'drop trigger ss_aud_trig'; exception when others then null; end;
/

begin execute immediate 'drop trigger schedule_settings_trigger'; exception when others then null; end;
/


declare
   c number;
begin

   select count(*)
   into   c
   from   user_tables
   where  table_name = 'SCHEDULE_SETTINGS';

   if c = 0 then   
      execute immediate '
CREATE TABLE SCHEDULE_SETTINGS (
                            ID          NUMBER                                             CONSTRAINT NN_SS_ID NOT NULL
                         ,NAME    VARCHAR2(50)                                           CONSTRAINT NN_SS_NAME NOT NULL
                  ,DESCRIPTION   VARCHAR2(500)                                    CONSTRAINT NN_SS_DESCRIPTION NOT NULL
                    ,DATA_TYPE    VARCHAR2(10)                                      CONSTRAINT NN_SS_DATA_TYPE NOT NULL
                        ,VALUE   VARCHAR2(100)                                          CONSTRAINT NN_SS_VALUE NOT NULL
)
tablespace &data_ts';

      execute immediate 'alter table schedule_settings add constraint pk_ss primary key (id) using index tablespace &ind_ts';
      execute immediate 'create unique index ss_n on  schedule_settings (name) tablespace &ind_ts';      
      
   end if;
   
-- Remove new settings added as part of this patch in case re-running ...
   execute immediate 'delete schedule_settings where id in (-41,-42,-43,-44,-45,-46,-47,-48,-49)';
   
   execute immediate 'select count(*) from schedule_settings' into c;
   
   if c = 0 then

      select count(*)
      into   c
      from   user_tables
      where  table_name = 'SCHEDULE_SETTINGS_BKP';

      if c = 0 then
      -- create backup table
         execute immediate '  
      create table SCHEDULE_SETTINGS_BKP
      tablespace &data_ts
      as
      select *
      from   parameters
      where  param_id < 0';
      end if;

      execute immediate 'select count(*) from SCHEDULE_SETTINGS_BKP' into c;

      if c <> 40 then
         raise_application_error(-20000, 'SCHEDULE_SETTINGS_BKP has incorrect number of values - expected 40, found '||to_char(c)||'!');
      end if;
   
      execute immediate 'insert into SCHEDULE_SETTINGS
select param_id, name, description, data_type, value
from   SCHEDULE_SETTINGS_BKP';

      execute immediate 'select count(*) from schedule_settings' into c;   
      
   end if;

   if c <> 40 then
      raise_application_error(-20000, 'SCHEDULE_SETTINGS has incorrect number of values - expected 40, found '||to_char(c)||'!');
   end if;

-- If here, all good to wipe parameters ...
   delete parameters where param_id < 0;

   commit;
   
end;
/

insert into schedule_settings (ID, NAME, DESCRIPTION, DATA_TYPE, VALUE)
values (-41, 'SUDO_EXE_DIRECTORY', 'Location of the sudo binary. Only used if you intend to set SUDO_CREDENTIALS.'
||chr(10)||chr(10)||
'Change Effective: Immediate.', 'CHARACTER', '/usr/bin');

insert into schedule_settings (ID, NAME, DESCRIPTION, DATA_TYPE, VALUE)
values (-42, 'SUDO_CREDENTIALS', 'User to sudo as when running all host commands. Must be set to <NONE> to not use sudo. This feature requires the named user to have a passwordless entry in the sudoers file on the DB server in the format: oracle ALL=(user) NOPASSWD: ALL'
||chr(10)||chr(10)||
'Change Effective: Immediate.', 'CHARACTER', '<NONE>');

insert into schedule_settings (ID, NAME, DESCRIPTION, DATA_TYPE, VALUE)
values (-43, 'VIEW_SOURCE_LEVEL', 'Minimum role level required to view source code through the console.'
||chr(10)||chr(10)||
'Change Effective: Immediate.', 'CHARACTER', 'CRONOLOGY_READONLY');

insert into schedule_settings (ID, NAME, DESCRIPTION, DATA_TYPE, VALUE)
values (-44, 'CONSOLE_2FA', 'Use Two Factor Authentication for console logins.'
||chr(10)||chr(10)||
'Change Effective: Immediate.', 'CHARACTER', 'OFF');

insert into schedule_settings (ID, NAME, DESCRIPTION, DATA_TYPE, VALUE)
values (-45, 'CONSOLE_2FA_VERFICATION_WINDOW', 'The number of 30 second time steps (+/-) to use when verifying TOTP codes.'
||chr(10)||chr(10)||
'Change Effective: Immediate.', 'NUMBER', '1');

insert into schedule_settings (ID, NAME, DESCRIPTION, DATA_TYPE, VALUE)
values (-46, 'CONSOLE_MAX_SESSION_MINS', 'The maximum session length in minutes for a console session, after which the user is logged off. Use 0 for no timeout.'
||chr(10)||chr(10)||
'Change Effective: Immediate.', 'NUMBER', '0');

insert into schedule_settings (ID, NAME, DESCRIPTION, DATA_TYPE, VALUE)
values (-47, 'EMAIL_SETTINGS_PART_4', 'Use STARTTLS. Use the CRONOLOGY.ADMIN.SET_SMTP_SETTINGS procedure to set this parameter.'
||chr(10)||chr(10)||
'Change Effective: Immediate.', 'CHARACTER', 'OFF');

insert into schedule_settings (ID, NAME, DESCRIPTION, DATA_TYPE, VALUE)
values (-48, 'WALLET_DIRECTORY', 'Location of an Oracle Wallet containing authentication credemtials / certificates. Use the CRONOLOGY.ADMIN.SET_WALLET procedure to set this parameter.'
||chr(10)||chr(10)||
'Change Effective: Immediate.', 'CHARACTER', '<NONE>');

insert into schedule_settings (ID, NAME, DESCRIPTION, DATA_TYPE, VALUE)
values (-49, 'WALLET_PASSWORD', 'Oracle Wallet password (if applicable). Use the CRONOLOGY.ADMIN.SET_WALLET procedure to set this parameter.'
||chr(10)||chr(10)||
-- <NONE>
'Change Effective: Immediate.', 'CHARACTER', 'DCCE47BD535D4A45EA35E117D18F079EC07AC26849171395F1D03DE64053FB403EBD1142E25C83BFCE5FCCAE80E50113');


update audit_history
set    id_type = 'S',
       table_name = 'SCHEDULE_SETTINGS',
       audit_detail = replace(audit_detail,'{"PARAM_ID":','{"ID":')
where id_type = 'P'
and   id < 0;

commit;

-------------------------------
-- NEW SCHEDULE_SETTINGS END --
-------------------------------

prompt

prompt Compiling packages ...
prompt

prompt INSTALL_INFO ...
@install_info.plb
prompt CRYPTO ...
@crypto.plb
prompt INTERNAL ...
@internal.plb
prompt EXTERNAL ...
@external.plb
prompt API ...
@api.plb
prompt ADMIN ...
@admin.plb

prompt TRIGGERS ...
@triggers.sql

set define '&'

begin
   crypto.update_encryption;
   commit;
end;
/

-- Align any 4.1 status entries with 4.2 format
begin
   for u in (select rowid rid, case when status like 'RUNNING (started%'           then 'RUNNING'
                                    when status like 'STOPPED __-___-____%FAILED%' then 'STOPPED (FAILED)'
                                    when status like 'STOPPED __-___-____%'        then 'STOPPED'
                               else
                                  status
                               end new_status
             from bg_process_history) loop
      update bg_process_history
      set    status = u.new_status
      where  rowid = u.rid;
   end loop;
   commit;
end;
/

update bg_process_history
set    status = case when info = 'Delayed stop requested by user.' then 'RUNNING' else 'RUNNING (DELAYED STOP PENDING)' end,
       info   = case when info = 'Delayed stop set.' then 'Delayed stop set: '||substr(status,19,17)
                else info
                end
where  process = 'JPLR'
and    status like 'RUNNING (stop due __-___-____ __:__)';

commit;

begin execute immediate 'ALTER TABLE JOB drop CONSTRAINT CK_EMAIL_LOG'; exception when others then null; end;
/

PROMPT Altering JOB adding CK_EMAIL_LOG
ALTER TABLE JOB
ADD CONSTRAINT CK_EMAIL_LOG
CHECK
((email is null and email_log = 'N') or (email is not null and email_log in ('Y','N','M','O')))
/

begin execute immediate 'alter table job_status_overrides drop constraint ck_jso'; exception when others then null; end;
/

PROMPT Altering JOB_STATUS_OVERRIDES adding CK_JSO
ALTER TABLE JOB_STATUS_OVERRIDES
ADD CONSTRAINT CK_JSO
CHECK
( match in ('Y','N') and mark_as in ('COMPLETE','FAILED','COMPLETE*'))
/

PROMPT Creating View JOB_V
CREATE OR REPLACE FORCE VIEW job_v as
select  /*+ leading (j jrh) use_nl_with_index(jrh pk_jrunhist) */
        j.job_id               ,
        j.parent_job_id        ,
        level                j_level,
        j.job_name             ,
        j.next_sub_dttm        ,
        j.last_sub_dttm        ,
        j.avg_duration         ,
        case
           when runtime_warning is not null then
              to_char(extract(day from numtodsinterval(runtime_warning,'DAY')),'FM99900')||':'||substr(numtodsinterval(runtime_warning,'DAY'),12,8)
           else
              null
        end        runtime_warning_disp,
       (select jst.description from job_source_types jst where jst.source_type = j.source_type) source_type,
        j.schedule_type, j.job_cron_str, j.ssh_conn_string, j.source_name, j.source_dir, j.db_conn_string, j.username,
        case
           when j.email_log = 'Y' then 'Job Log'
           when j.email_log = 'O' then 'Job Log *'
           when j.email_log = 'N' and j.email is not null then '<NONE>'
           when j.email_log = 'M' then 'Message'
        end  display_email_log      ,
        j.email,
        j.email_on_failure,
        j.email_on_failure_cc_ops,
        case when j.schedule_type = 'TIMED' then
           case when j.exe_window_start_dttm is not null and j.exe_window_end_dttm is not null then '>>   ' end ||
           case when j.next_sub_dttm >= to_date('31-DEC-4712','DD-MON-YYYY') then 'JOB WILL NOT RUN AGAIN' else to_char(j.next_sub_dttm,'DY DD-MON-YYYY HH24:MI:SS') end ||
           case when j.exe_window_start_dttm is not null and j.exe_window_end_dttm is not null then '   <<' end
        end display_next_sub_dttm,
       jrh.SUB_DTTM           ,
       jrh.START_DTTM         ,
       jrh.END_DTTM           ,
       case
          when jrh.duration is not null then
             jrh.duration
          when jrh.job_id is null then
          -- job never run
             null
          else
            ltrim(
             case when jrh.status in ('WAITING','SUSPENDED') then
                to_char(extract(day from systimestamp - cast(jrh.wait_start_dttm as timestamp)),'FM99900')||':'||substr(systimestamp - cast(jrh.wait_start_dttm as timestamp),12,8)
             else
                to_char(extract(day from cast( coalesce(jrh.end_dttm,sysdate) as timestamp)- cast(jrh.start_dttm as timestamp)),'FM99900')||':'||substr(cast( coalesce(jrh.end_dttm,sysdate) as timestamp) - cast(jrh.start_dttm as timestamp),12,8)
             end,':')
       end duration,
       jrh.STATUS             ,
       jrh.RUN_STAND_ALONE    ,
       jrh.OK_TO_CONTINUE     ,
       jrh.USE_OVERRIDE_PARAMS,
       jrh.ACTION_USERNAME    ,
       jrh.INFO,
       jrh.PROCESS,
       case
          when connect_by_root   j.job_frequency like '%+' then
              ltrim(substr(connect_by_root   j.job_frequency,1,3),'0')||' '||
              replace(substr(connect_by_root   j.job_frequency,4),'+',case when substr(connect_by_root   j.job_frequency,1,3) = '001' then '' else 'S' end)
          else
             connect_by_root   j.job_frequency
       end                        job_frequency,
       prior j.job_name  parent_job_name,
       j.source_type source_type_short,
       j.job_description,
       j.doc_link,
       case
          when (trunc(connect_by_root j.next_sub_dttm) = trunc(sysdate)) or jrh.ok_to_continue = 'N' then 'Y'
       end today_only
from    job j
left outer join job_run_history jrh
on      j.job_id = jrh.job_id
and     j.last_sub_dttm = jrh.sub_dttm
where   j.job_online         = 'Y'
start   with j.next_sub_dttm > to_date('01-JAN-4712BC','DD-MON-YYYYBC')
connect by prior j.job_id    = j.parent_job_id
order   siblings by j.next_sub_dttm, j.job_name
/

begin execute immediate 'ALTER TABLE PARAMETERS drop CONSTRAINT CK_PARAM'; exception when others then null; end;
/

PROMPT Altering PARAMETERS adding CK_PARAM
ALTER TABLE PARAMETERS
ADD CONSTRAINT CK_PARAM
CHECK
(
 ((type = 'SQL'    and source is not null and value is null) or
  (type = 'FIXED'  and source is null     and value is not null) or
  (type = 'ENVVAR' and source is null     and value is not null and value like '%$_%'))
 and
 (data_type in ('NUMBER','CHARACTER','DATE','BOOLEAN','TIMESTAMP'))
  and
 (conceal_values in ('Y','N'))
)
/

PROMPT Altering JOB increasing EMAIL length
alter table job modify (email varchar(500));

PROMPT Altering JOB increasing EMAIL_ON_FAILURE length
alter table job modify (email_on_failure varchar(500));


----------------------------------------------------------------------------------------------------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------

set feed off
prompt
prompt Recompiling Invalid objects ...
prompt ===============================
prompt
BEGIN
   dbms_utility.compile_schema(user,true);
   DBMS_SESSION.RESET_PACKAGE;
END;
/
prompt Complete.

prompt
prompt Recreating JAVA_LIB ...
prompt =======================
prompt

delete java_lib_class;
insert into java_lib_class values ('CAFEBABE0000003200F20A001000730800740800750A007600770A007800790A0078007A07007B0A0046007C0A0046007D07');
update java_lib_class set bytecode = bytecode || '007E08007F0800800A008100820A008100830800840700850A008600870A008800890A000A008A08008B0A000A008C0A0081';
update java_lib_class set bytecode = bytecode || '008D08008E08008F0700900A001900910800920700930700940A008600950A001D00960A001C00970A001C00980800990A00';
update java_lib_class set bytecode = bytecode || '19009A0A001C009B0A0019009C0A0086009D07009E0A0086009F0A00A0009B0A00A1009B0A008600A20700A30A0019007308';
update java_lib_class set bytecode = bytecode || '00A40A001900A50800A60A00A700A80800A90800AA0800AB0800AC0700AD0A003600910A003600AE09004600AF09004600B0';
update java_lib_class set bytecode = bytecode || '0A003600B10A003600B20A003600B30A003600B40A003600B50700B60A004000910A004000B70A0040009B0A004600B80A00';
update java_lib_class set bytecode = bytecode || '4600B90700BA01000753554343455353010001490100074641494C5552450100063C696E69743E010003282956010004436F';
update java_lib_class set bytecode = bytecode || '646501000F4C696E654E756D6265725461626C6501000A5061636B61676556657201001428294C6A6176612F6C616E672F53';
update java_lib_class set bytecode = bytecode || '7472696E673B01000C537472696E67546F436C6F62010025284C6A6176612F6C616E672F537472696E673B294C6F7261636C';
update java_lib_class set bytecode = bytecode || '652F73716C2F434C4F423B01000D537461636B4D61705461626C6507007B010011457865634F53436F6D6D616E64436C6F62';
update java_lib_class set bytecode = bytecode || '01007F284C6A6176612F6C616E672F537472696E673B4C6A6176612F6C616E672F537472696E673B4C6A6176612F6C616E67';
update java_lib_class set bytecode = bytecode || '2F537472696E673B4C6A6176612F6C616E672F537472696E673B4C6A6176612F6C616E672F537472696E673B4C6A6176612F';
update java_lib_class set bytecode = bytecode || '6C616E672F537472696E673B294C6F7261636C652F73716C2F434C4F423B01000D457865634F53436F6D6D616E6401008028';
update java_lib_class set bytecode = bytecode || '4C6A6176612F6C616E672F537472696E673B4C6A6176612F6C616E672F537472696E673B4C6A6176612F6C616E672F537472';
update java_lib_class set bytecode = bytecode || '696E673B4C6A6176612F6C616E672F537472696E673B4C6A6176612F6C616E672F537472696E673B4C6A6176612F6C616E67';
update java_lib_class set bytecode = bytecode || '2F537472696E673B294C6A6176612F6C616E672F537472696E673B07007E0700BB0700930700900700BC0700BD07009E0700';
update java_lib_class set bytecode = bytecode || 'A30700A30700BE0700BE01000C4765744F5344657461696C7301000E4765744A61766156657273696F6E01000A46696C6545';
update java_lib_class set bytecode = bytecode || '7869737473010015284C6A6176612F6C616E672F537472696E673B29490700AD01000F46696C6549734469726563746F7279';
update java_lib_class set bytecode = bytecode || '01000A46696C65497346696C6501000A46696C6549735069706501000A46696C6544656C65746501001146696C6545786563';
update java_lib_class set bytecode = bytecode || '4F53436F6D6D616E64010092284C6A6176612F6C616E672F537472696E673B4C6A6176612F6C616E672F537472696E673B4C';
update java_lib_class set bytecode = bytecode || '6A6176612F6C616E672F537472696E673B4C6A6176612F6C616E672F537472696E673B4C6A6176612F6C616E672F53747269';
update java_lib_class set bytecode = bytecode || '6E673B4C6A6176612F6C616E672F537472696E673B4C6A6176612F6C616E672F537472696E673B294C6A6176612F6C616E67';
update java_lib_class set bytecode = bytecode || '2F537472696E673B01001546696C65457865634F53436F6D6D616E64436C6F62010091284C6A6176612F6C616E672F537472';
update java_lib_class set bytecode = bytecode || '696E673B4C6A6176612F6C616E672F537472696E673B4C6A6176612F6C616E672F537472696E673B4C6A6176612F6C616E67';
update java_lib_class set bytecode = bytecode || '2F537472696E673B4C6A6176612F6C616E672F537472696E673B4C6A6176612F6C616E672F537472696E673B4C6A6176612F';
update java_lib_class set bytecode = bytecode || '6C616E672F537472696E673B294C6F7261636C652F73716C2F434C4F423B0100083C636C696E69743E01000A536F75726365';
update java_lib_class set bytecode = bytecode || '46696C6501000D4A4156415F4C49422E6A6176610C004A004B010010245265766973696F6E3A2023313420240100176A6462';
update java_lib_class set bytecode = bytecode || '633A64656661756C743A636F6E6E656374696F6E0700BF0C00C000C10700C20C00C300C40C00C500C60100156A6176612F73';
update java_lib_class set bytecode = bytecode || '716C2F53514C457863657074696F6E0C005600570C005000510100106A6176612F6C616E672F537472696E670100022D6301';
update java_lib_class set bytecode = bytecode || '00022D750700BD0C00C700C80C00C900CA010004253033640100106A6176612F6C616E672F4F626A6563740700BB0C00CB00';
update java_lib_class set bytecode = bytecode || 'CC0700CD0C00CE00CF0C00D000D10100024F4E0C00D200D30C00D4004B010001590100033030300100176A6176612F6C616E';
update java_lib_class set bytecode = bytecode || '672F537472696E674275696C6465720C004A00D50100000100166A6176612F696F2F42756666657265645265616465720100';
update java_lib_class set bytecode = bytecode || '196A6176612F696F2F496E70757453747265616D5265616465720C00D600D70C004A00D80C004A00D90C00DA004F0100010A';
update java_lib_class set bytecode = bytecode || '0C00DB00DC0C00DD004B0C00DE004F0C00DF00D70100136A6176612F696F2F494F457863657074696F6E0C00E000E10700E2';
update java_lib_class set bytecode = bytecode || '0700E30C00E4004B0100136A6176612F6C616E672F457863657074696F6E0100044A52450A0C00DB00E50100076F732E6E61';
update java_lib_class set bytecode = bytecode || '6D650700E60C00E700E80100012001000A6F732E76657273696F6E0100076F732E6172636801000C6A6176612E7665727369';
update java_lib_class set bytecode = bytecode || '6F6E01000C6A6176612F696F2F46696C650C00E900EA0C004700480C004900480C00EB00EA0C00EC00EA0C00ED00EA0C00EE';
update java_lib_class set bytecode = bytecode || '00EA0C00EF00F00100126A6176612F696F2F46696C655772697465720C00F100D50C006B00660C006C006D0100084A415641';
update java_lib_class set bytecode = bytecode || '5F4C49420100116A6176612F6C616E672F50726F636573730100135B4C6A6176612F6C616E672F537472696E673B0100116A';
update java_lib_class set bytecode = bytecode || '6176612F6C616E672F52756E74696D650100136A6176612F6C616E672F5468726F7761626C650100166A6176612F73716C2F';
update java_lib_class set bytecode = bytecode || '4472697665724D616E6167657201000D676574436F6E6E656374696F6E010029284C6A6176612F6C616E672F537472696E67';
update java_lib_class set bytecode = bytecode || '3B294C6A6176612F73716C2F436F6E6E656374696F6E3B01000F6F7261636C652F73716C2F434C4F4201000F637265617465';
update java_lib_class set bytecode = bytecode || '54656D706F7261727901002A284C6A6176612F73716C2F436F6E6E656374696F6E3B5A49294C6F7261636C652F73716C2F43';
update java_lib_class set bytecode = bytecode || '4C4F423B010009736574537472696E67010016284A4C6A6176612F6C616E672F537472696E673B294901000A67657452756E';
update java_lib_class set bytecode = bytecode || '74696D6501001528294C6A6176612F6C616E672F52756E74696D653B0100046578656301003B285B4C6A6176612F6C616E67';
update java_lib_class set bytecode = bytecode || '2F537472696E673B5B4C6A6176612F6C616E672F537472696E673B294C6A6176612F6C616E672F50726F636573733B010007';
update java_lib_class set bytecode = bytecode || '77616974466F720100032829490100116A6176612F6C616E672F496E746567657201000776616C75654F660100162849294C';
update java_lib_class set bytecode = bytecode || '6A6176612F6C616E672F496E74656765723B010006666F726D6174010039284C6A6176612F6C616E672F537472696E673B5B';
update java_lib_class set bytecode = bytecode || '4C6A6176612F6C616E672F4F626A6563743B294C6A6176612F6C616E672F537472696E673B010006657175616C7301001528';
update java_lib_class set bytecode = bytecode || '4C6A6176612F6C616E672F4F626A6563743B295A0100026763010015284C6A6176612F6C616E672F537472696E673B295601';
update java_lib_class set bytecode = bytecode || '000E676574496E70757453747265616D01001728294C6A6176612F696F2F496E70757453747265616D3B010018284C6A6176';
update java_lib_class set bytecode = bytecode || '612F696F2F496E70757453747265616D3B2956010013284C6A6176612F696F2F5265616465723B2956010008726561644C69';
update java_lib_class set bytecode = bytecode || '6E65010006617070656E6401002D284C6A6176612F6C616E672F537472696E673B294C6A6176612F6C616E672F537472696E';
update java_lib_class set bytecode = bytecode || '674275696C6465723B010005636C6F7365010008746F537472696E6701000E6765744572726F7253747265616D01000F6765';
update java_lib_class set bytecode = bytecode || '744F757470757453747265616D01001828294C6A6176612F696F2F4F757470757453747265616D3B0100146A6176612F696F';
update java_lib_class set bytecode = bytecode || '2F4F757470757453747265616D0100136A6176612F696F2F496E70757453747265616D01000764657374726F7901002D284C';
update java_lib_class set bytecode = bytecode || '6A6176612F6C616E672F4F626A6563743B294C6A6176612F6C616E672F537472696E674275696C6465723B0100106A617661';
update java_lib_class set bytecode = bytecode || '2F6C616E672F53797374656D01000B67657450726F7065727479010026284C6A6176612F6C616E672F537472696E673B294C';
update java_lib_class set bytecode = bytecode || '6A6176612F6C616E672F537472696E673B01000665786973747301000328295A01000B69734469726563746F727901000669';
update java_lib_class set bytecode = bytecode || '7346696C6501000664656C65746501000D6372656174654E657746696C6501000D73657445786563757461626C6501000528';
update java_lib_class set bytecode = bytecode || '5A5A295A010005777269746500210046001000000002000A004700480000000A004900480000000F0001004A004B0001004C';
update java_lib_class set bytecode = bytecode || '0000001D00010001000000052AB70001B100000001004D0000000600010000000D0009004E004F0001004C0000001B000100';
update java_lib_class set bytecode = bytecode || '00000000031202B000000001004D000000060001000000140009005000510001004C0000005A000400030000001A1203B800';
update java_lib_class set bytecode = bytecode || '044C2B03100CB800054D2C0A2AB60006572CB04C01B0000100000016001700070002004D0000001A00060000001C0006001D';
update java_lib_class set bytecode = bytecode || '000E001E0015001F00170021001800230052000000060001570700530009005400550001004C0000002F0006000700000013';
update java_lib_class set bytecode = bytecode || '2A2B2C2D19041905B800083A061906B80009B000000001004D0000000A000200000029000D002A0009005600570001004C00';
update java_lib_class set bytecode = bytecode || '0004EE0005001100000236013A06013A07013A08013A0903BD000A3A0A1905C7001906BD000A59032A535904120B5359052B';
update java_lib_class set bytecode = bytecode || '533A0BA700261006BD000A59031904535904120C53590519055359062A535907120B5359082B533A0BB8000D3A0C190C190B';
update java_lib_class set bytecode = bytecode || '190AB6000E3A06120F04BD001059031906B60011B8001253B800133A092D1214B60015990008190CB600162C1217B600159A';
update java_lib_class set bytecode = bytecode || '000D19091218B600159A00D9BB0019591909B7001A3A08121B3A0D19091218B60015990042BB001C59BB001D591906B6001E';
update java_lib_class set bytecode = bytecode || 'B7001FB700203A071907B60021593A0DC6001319081222B60023190DB6002357A7FFE81907B600241908B600253A09A70081';
update java_lib_class set bytecode = bytecode || 'BB001C59BB001D591906B60026B7001FB700203A071907B60021593A0DC6001319081222B60023190DB6002357A7FFE81907';
update java_lib_class set bytecode = bytecode || 'B600241908B600251909B60015990038BB001C59BB001D591906B6001EB7001FB700203A071907B60021593A0DC600131908';
update java_lib_class set bytecode = bytecode || '1222B60023190DB6002357A7FFE81907B600241908B600253A0919093A0D1907C6000D1907B60024A700053A0E1906C60025';
update java_lib_class set bytecode = bytecode || '1906B60028B600291906B60026B6002A1906B6001EB6002AA700053A0E1906B6002B190DB03A0ABB001959B7002D122EB600';
update java_lib_class set bytecode = bytecode || '23190AB6002FB600253A0B1907C6000D1907B60024A700053A0C1906C600251906B60028B600291906B60026B6002A1906B6';
update java_lib_class set bytecode = bytecode || '001EB6002AA700053A0C1906B6002B190BB03A0F1907C6000D1907B60024A700053A101906C600251906B60028B600291906';
update java_lib_class set bytecode = bytecode || 'B60026B6002A1906B6001EB6002AA700053A101906B6002B190FBF000A0176017B017E00270185019D01A00027000C017101';
update java_lib_class set bytecode = bytecode || 'AA002C01C701CC01CF002701D601EE01F10027000C017101FB000001AA01C201FB000002020207020A002702110229022C00';
update java_lib_class set bytecode = bytecode || '2701FB01FD01FB00000002004D0000011200440000002F000300300006003100090032000C00360012003800170039002D00';
update java_lib_class set bytecode = bytecode || '3B0050003D0055003E0060004000760041007F0042008400440097004700A2004800A6004900B0004B00C5004C00D0004D00';
update java_lib_class set bytecode = bytecode || 'E0004F00E5005000EF005301040054010F0055011F0057012400580131005B0146005C0151005D0161005F01660061016D00';
update java_lib_class set bytecode = bytecode || '640171006D0176006F017B0072017E00700180007401850076018D007701950078019D007B01A0007901A2007C01AA006601';
update java_lib_class set bytecode = bytecode || 'AC006801C2006D01C7006F01CC007201CF007001D1007401D6007601DE007701E6007801EE007B01F1007901F3007C01FB00';
update java_lib_class set bytecode = bytecode || '6D0202006F02070072020A0070020C00740211007602190077022100780229007B022C0079022E007C00520000013E001EFF';
update java_lib_class set bytecode = bytecode || '002D000B07005807005807005807005807005807005807005907005A07005B07005807005C0000FC002207005CFC00330700';
update java_lib_class set bytecode = bytecode || '5D12FC002D0700581A0E141A261A04FA0006FF0010000E07005807005807005807005807005807005807005907005A07005B';
update java_lib_class set bytecode = bytecode || '07005807005C07005C07005D070058000107005E015F07005E0104FF0002000A070058070058070058070058070058070058';
update java_lib_class set bytecode = bytecode || '07005907005A07005B070058000107005FFF0024000C07005807005807005807005807005807005807005907005A07005B07';
update java_lib_class set bytecode = bytecode || '0058070060070058000107005E015F07005E0104FF0002000A07005807005807005807005807005807005807005907005A07';
update java_lib_class set bytecode = bytecode || '005B0700580001070061FF000E001007005807005807005807005807005807005807005907005A07005B0700580000000000';
update java_lib_class set bytecode = bytecode || '070062000107005E015F07005E010400090063004F0001004C00000045000200000000002DBB001959B7002D1230B80031B6';
update java_lib_class set bytecode = bytecode || '00231232B600231233B80031B600231232B600231234B80031B60023B60025B000000001004D000000060001000000820009';
update java_lib_class set bytecode = bytecode || '0064004F0001004C0000001E00010000000000061235B80031B000000001004D000000060001000000860009006500660001';
update java_lib_class set bytecode = bytecode || '004C0000004A0003000200000018BB0036592AB700374C2BB60038990007B20039ACB2003AAC00000002004D000000120004';
update java_lib_class set bytecode = bytecode || '0000008A0009008B0010008C0014008E0052000000080001FC00140700670009006800660001004C0000004A000300020000';
update java_lib_class set bytecode = bytecode || '0018BB0036592AB700374C2BB6003B990007B20039ACB2003AAC00000002004D000000120004000000920009009300100094';
update java_lib_class set bytecode = bytecode || '001400960052000000080001FC00140700670009006900660001004C0000004A0003000200000018BB0036592AB700374C2B';
update java_lib_class set bytecode = bytecode || 'B6003C990007B20039ACB2003AAC00000002004D0000001200040000009A0009009B0010009C0014009E0052000000080001';
update java_lib_class set bytecode = bytecode || 'FC00140700670009006A00660001004C000000590003000200000026BB0036592AB700374C2BB600389900112BB6003B9A00';
update java_lib_class set bytecode = bytecode || '0A2BB6003C990007B2003AACB20039AC00000002004D000000120004000000A2000900A3001E00A7002200A9005200000009';
update java_lib_class set bytecode = bytecode || '0002FC001E070067030009006B00660001004C0000005E0003000200000023BB0036592AB700374C2BB600389900122BB600';
update java_lib_class set bytecode = bytecode || '3D990007B20039ACB2003AACB20039AC00000002004D0000001A0006000000AD000900AF001000B0001700B1001B00B3001F';
update java_lib_class set bytecode = bytecode || '00B60052000000090002FC001B070067030009006C006D0001004C000000F90006000C00000083BB0036592AB700373A0719';
update java_lib_class set bytecode = bytecode || '07B6003E5719070403B6003F57BB0040592AB700413A081908BB001959B7002D2CB600231222B60023B60025B600421908B6';
update java_lib_class set bytecode = bytecode || '00432B2A2D190419051906B800083A0919093A0A2AB8004457190AB03A07BB001959B7002D122EB600231907B6002FB60025';
update java_lib_class set bytecode = bytecode || '3A082AB80044571908B03A0B2AB8004457190BBF0004000000510059002C000000510079000000590071007900000079007B';
update java_lib_class set bytecode = bytecode || '007900000002004D00000032000C000000BD000A00BE001000C2001800C3002200C6003A00C7003F00C9004D00CA005100D3';
update java_lib_class set bytecode = bytecode || '005900CC005B00CE007100D300520000000C0002F7005907005F5F0700610009006E006F0001004C00000031000700080000';
update java_lib_class set bytecode = bytecode || '00152A2B2C2D190419051906B800453A071907B80009B000000001004D0000000A0002000000D9000F00DA00080070004B00';
update java_lib_class set bytecode = bytecode || '01004C00000025000100000000000903B3003904B3003AB100000001004D0000000A00020000001000040011000100710000';
update java_lib_class set bytecode = bytecode || '00020072';

commit;
begin
   admin.reload_java_lib;
end;
/
prompt Complete.

set serveroutput on lines 32767

prompt
prompt Validating Patch ...
prompt ====================
prompt
declare
   ins_status clob;
begin
   install_info.install_valid_metadata;
   ins_status := install_info.validate_install;
   dbms_output.put_line(ins_status);
   if ins_status <> 'Installation valid.' then
      dbms_output.put_line(chr(10));
      ins_status := install_info.show_invalid;
      dbms_output.put_line(ins_status);
      raise_application_error(-20000,'Installation invalid.');
   end if;
end;
/

-- patch complete - safe to drop sys parameters backup 
begin execute immediate 'drop table SCHEDULE_SETTINGS_BKP purge'; exception when others then null; end;
/

set pages 0 feed off
select chr(10)||message||chr(10)||rpad('=',length(message),'=')
from (select 'Cronology Server Patch &iv Complete on '||
             upper(substr(host_name,1,decode(instr(host_name,'.'),0,9999,instr(host_name,'.')-1)))||' '||upper(instance_name) message
      from   v$instance);

spool off

prompt
exit
