create or replace trigger after_db_startup
after startup on database 
begin
   internal.db_start_trigger;
end;
/

create or replace trigger before_db_shutdown
before shutdown on database
begin
   internal.db_stop_trigger;
end;
/

create or replace trigger job_trigger
before insert or update of password, job_id, job_description, completion_message, job_frequency_sql,  msg_text on job
for each row
begin
   :new.job_description := replace(:new.job_description,chr(13),'');
   :new.completion_message := replace(:new.completion_message,chr(13),'');
   :new.job_frequency_sql := replace(:new.job_frequency_sql,chr(13),'');
   :new.msg_text := replace(:new.msg_text,chr(13),'');
   internal.job_trigger(:new.password, :new.job_id, :old.job_id);
end;
/

create or replace trigger parameters_trigger
before insert or update of value, override_value, description on parameters
for each row
begin
   if updating or inserting then
      :new.value := replace(:new.value,chr(13),'');
      :new.description := replace(:new.description,chr(13),'');
      :new.source := replace(:new.source,chr(13),'');      
   end if;
end;
/

create or replace trigger schedule_settings_trigger
before insert or 
        update of value, description or
        delete on schedule_settings
for each row
declare
   dummy_out  varchar2(1);
begin
   if updating or inserting then
      :new.value := replace(:new.value,chr(13),'');
      :new.description := replace(:new.description,chr(13),'');
      internal.schedule_settings_trigger (:new.name, :new.value, :new.data_type);
   elsif deleting then
      internal.schedule_settings_trigger (null, dummy_out, null);
   end if;
end;
/

--------------------------------------------------------------------------------------------------------------------------------
-- no need to use FOLLOWS syntax for ordering, the 2 triggers above are BEFORE FOR EACH ROW
-- the compound triggers are BEFORE STATEMENT, AFTER EACH ROW, AFTER STATEMENT i.e. there is no clash of timing points
-- the auditing AFTER EACH ROW will run always run after the BEFORE EACH ROW (validation/column changes) in the above triggers
--------------------------------------------------------------------------------------------------------------------------------

CREATE OR REPLACE TRIGGER p_aud_trig for insert
                                          or 
                                          update
                                          or
                                          delete on parameters COMPOUND TRIGGER
                                 rid_tab           internal.rid_tab;
                                 l_scn             number;
                                 l_table_name varchar2(30) := 'PARAMETERS';
                                 l_audit      boolean;

   BEFORE STATEMENT IS
   BEGIN
      if deleting then
         l_scn := DBMS_FLASHBACK.GET_SYSTEM_CHANGE_NUMBER;
      end if;
      l_audit := crypto.get_param('AUDIT_CONTROL') = 'ON' and internal.c_current_user <> crypto.upper_app_schema;      
   END BEFORE STATEMENT;
   
   AFTER EACH ROW IS
      changes_detected boolean := false;
      l_json varchar2(32767);
      l_conceal boolean := false;
   BEGIN
      if l_audit then
         case when updating then
            internal.aud_upd_col (l_json, internal.json_str(:old.param_id), internal.json_str(:new.param_id),'PARAM_ID' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.name), internal.json_str(:new.name),'NAME' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.description), internal.json_str(:new.description),'DESCRIPTION' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.data_type), internal.json_str(:new.data_type),'DATA_TYPE' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.type), internal.json_str(:new.type),'TYPE' , changes_detected, false);
         -- This is the only column in all the auditing that may pass conceal=true, value is a character column so expect results to be in double quotes
         -- always use new value of conceal_values - may lead to some values getting displayed if switching from Y to N
            if :new.conceal_values = 'Y' then
               l_conceal := true;
            end if;
            internal.aud_upd_col (l_json, internal.json_str(:old.value), internal.json_str(:new.value),'VALUE' , changes_detected, false, null, l_conceal);
            internal.aud_upd_col (l_json, internal.json_str(:old.source), internal.json_str(:new.source),'SOURCE' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.conceal_values), internal.json_str(:new.conceal_values),'CONCEAL_VALUES' , changes_detected, false);
            if changes_detected then
               internal.aud_upd_row('P',:old.param_id, l_table_name, substr(l_json,2));
            end if;
         when inserting then
            rid_tab(rid_tab.count+1) := :new.rowid;
         when deleting then
            rid_tab(rid_tab.count+1) := :old.rowid;
         end case;
      end if;
   END AFTER EACH ROW;

   AFTER STATEMENT IS
      l_sql  varchar2(4000) := q'!select 'P', t.param_id, '"PARAM_ID":'||internal.json_str(t.PARAM_ID)
||','||'"NAME":'||internal.json_str(t.NAME)
||','||'"DESCRIPTION":'||internal.json_str(t.DESCRIPTION)
||','||'"DATA_TYPE":'||internal.json_str(t.DATA_TYPE)
||','||'"TYPE":'||internal.json_str(t.TYPE)
||','||'"VALUE":'||internal.json_str(case when t.conceal_values = 'Y' then rpad('*',length(t.value),'*') else t.VALUE end)
||','||'"SOURCE":'||internal.json_str(t.SOURCE)
||','||'"CONCEAL_VALUES":'||internal.json_str(t.CONCEAL_VALUES)
from   parameters!';
   BEGIN
      if l_audit then
         case when inserting then
            if rid_tab.count > 0 then
               internal.aud_ins_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab);
            end if;
         when deleting then
            if rid_tab.count > 0 then
               internal.aud_del_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab, p_scn => l_scn);
            end if;
         when updating then
            null;
         end case;
      end if;
   END AFTER STATEMENT;
END;
/

CREATE OR REPLACE TRIGGER ss_aud_trig for insert
                                          or 
                                          update
                                          or
                                          delete on schedule_settings COMPOUND TRIGGER
                                 rid_tab           internal.rid_tab;
                                 l_scn             number;
                                 l_table_name varchar2(30) := 'SCHEDULE_SETTINGS';
                                 l_audit      boolean;

   BEFORE STATEMENT IS
   BEGIN
      if deleting then
         l_scn := DBMS_FLASHBACK.GET_SYSTEM_CHANGE_NUMBER;
      end if;
      l_audit := true;
   END BEFORE STATEMENT;
   
   AFTER EACH ROW IS
      changes_detected boolean := false;
      l_json varchar2(32767);
   BEGIN
      if l_audit then
         case when updating then
            internal.aud_upd_col (l_json, internal.json_str(:old.id), internal.json_str(:new.id),'ID' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.name), internal.json_str(:new.name),'NAME' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.description), internal.json_str(:new.description),'DESCRIPTION' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.value), internal.json_str(:new.value),'VALUE' , changes_detected, false);
            if changes_detected then
               internal.aud_upd_row('S',:old.id, l_table_name, substr(l_json,2));
            end if;
         when inserting then
            rid_tab(rid_tab.count+1) := :new.rowid;
         when deleting then
            rid_tab(rid_tab.count+1) := :old.rowid;
         end case;
      end if;
   END AFTER EACH ROW;

   AFTER STATEMENT IS
      l_sql  varchar2(4000) := q'!select 'S', t.id, '"ID":'||internal.json_str(t.ID)
||','||'"NAME":'||internal.json_str(t.NAME)
||','||'"DESCRIPTION":'||internal.json_str(t.DESCRIPTION)
||','||'"DATA_TYPE":'||internal.json_str(t.DATA_TYPE)
||','||'"VALUE":'||internal.json_str(t.VALUE)
from   schedule_settings!';
   BEGIN
      if l_audit then
         case when inserting then
            if rid_tab.count > 0 then
               internal.aud_ins_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab);
            end if;
         when deleting then
            if rid_tab.count > 0 then
               internal.aud_del_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab, p_scn => l_scn);
            end if;
         when updating then
            null;
         end case;
      end if;
   END AFTER STATEMENT;
END;
/

CREATE OR REPLACE TRIGGER jc_aud_trig for insert
                                          or 
                                          update
                                          or
                                          delete on job_calendar COMPOUND TRIGGER
                                 rid_tab           internal.rid_tab;
                                 l_scn             number;
                                 l_table_name varchar2(30) := 'JOB_CALENDAR';
                                 l_audit      boolean;

   BEFORE STATEMENT IS
   BEGIN
      if deleting then
         l_scn := DBMS_FLASHBACK.GET_SYSTEM_CHANGE_NUMBER;
      end if;
      l_audit := crypto.get_param('AUDIT_CONTROL') = 'ON' and internal.c_current_user <> crypto.upper_app_schema;      
   END BEFORE STATEMENT;

   AFTER EACH ROW IS
      changes_detected boolean := false;
      l_json varchar2(32767);
   BEGIN
      if l_audit then
         case when updating then
            internal.aud_upd_col (l_json, internal.json_str(:old.job_id),           internal.json_str(:new.job_id),'JOB_ID' , changes_detected, false, 'J');
            internal.aud_upd_col (l_json, internal.json_str(:old.run_dttm), internal.json_str(:new.run_dttm),'RUN_DTTM', changes_detected, false);
            if changes_detected then
               internal.aud_upd_row('J',:old.job_id,l_table_name, substr(l_json,2));
            end if;
         when inserting then
            rid_tab(rid_tab.count+1) := :new.rowid;
         when deleting then
            rid_tab(rid_tab.count+1) := :old.rowid;
         end case;
      end if;
   END AFTER EACH ROW;

   AFTER STATEMENT IS
      l_sql  varchar2(4000) := q'!select 'J', t.job_id, '"JOB_ID":'||internal.json_str(t.JOB_ID)
||','||'"JOB_NAME":'||(select internal.json_str(j.job_name) from job j where j.job_id = t.job_id)      
||','||'"RUN_DTTM":'||internal.json_str(t.RUN_DTTM)
from   job_calendar!';
   BEGIN
      if l_audit then
         case when inserting then
            if rid_tab.count > 0 then
               internal.aud_ins_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab);
            end if;
         when deleting then
            if rid_tab.count > 0 then
               internal.aud_del_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab, p_scn => l_scn);
            end if;
         when updating then
            null;
         end case;
      end if;
   END AFTER STATEMENT;
END;
/

CREATE OR REPLACE TRIGGER jd_aud_trig for insert
                                          or 
                                          update
                                          or
                                          delete on job_dependencies COMPOUND TRIGGER
                                 rid_tab           internal.rid_tab;
                                 l_scn             number;
                                 l_table_name varchar2(30) := 'JOB_DEPENDENCIES';
                                 l_audit      boolean;

   BEFORE STATEMENT IS
   BEGIN
      if deleting then
         l_scn := DBMS_FLASHBACK.GET_SYSTEM_CHANGE_NUMBER;
      end if;
      l_audit := crypto.get_param('AUDIT_CONTROL') = 'ON' and internal.c_current_user <> crypto.upper_app_schema;      
   END BEFORE STATEMENT;
   
   AFTER EACH ROW IS
      changes_detected boolean := false;
      l_json varchar2(32767);
   BEGIN
      if l_audit then
         case when updating then
            internal.aud_upd_col (l_json, internal.json_str(:old.job_id),           internal.json_str(:new.job_id),'JOB_ID' , changes_detected, false, 'J');
            internal.aud_upd_col (l_json, internal.json_str(:old.dependent_job_id), internal.json_str(:new.dependent_job_id),'DEPENDENT_JOB_ID', changes_detected, false, 'J');
            if changes_detected then
               internal.aud_upd_row('J',:old.job_id,l_table_name, substr(l_json,2));
            end if;
         when inserting then
            rid_tab(rid_tab.count+1) := :new.rowid;
         when deleting then
            rid_tab(rid_tab.count+1) := :old.rowid;
         end case;
      end if;
   END AFTER EACH ROW;

   AFTER STATEMENT IS
      l_sql  varchar2(4000) := q'!select 'J', t.job_id, '"JOB_ID":'||internal.json_str(t.JOB_ID)
||','||'"JOB_NAME":'||(select internal.json_str(j.job_name) from job j where j.job_id = t.job_id)
||','||'"DEPENDENT_JOB_ID":'||internal.json_str(t.DEPENDENT_JOB_ID)
||','||'"DEPENDENT_JOB_NAME":'||(select internal.json_str(j.job_name) from job j where j.job_id = t.dependent_job_id)
                                from   job_dependencies!';
   BEGIN
      if l_audit then
         case when inserting then
            if rid_tab.count > 0 then
               internal.aud_ins_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab);
            end if;
         when deleting then
            if rid_tab.count > 0 then
               internal.aud_del_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab, p_scn => l_scn);
            end if;
         when updating then
            null;
         end case;
      end if;
   END AFTER STATEMENT;
  
END;
/

CREATE OR REPLACE TRIGGER jpd_aud_trig for insert
                                          or 
                                          update
                                          or
                                          delete on job_param_dependencies COMPOUND TRIGGER
                                 rid_tab           internal.rid_tab;
                                 l_scn             number;
                                 l_table_name varchar2(30) := 'JOB_PARAM_DEPENDENCIES';
                                 l_audit      boolean;

   BEFORE STATEMENT IS
   BEGIN
      if deleting then
         l_scn := DBMS_FLASHBACK.GET_SYSTEM_CHANGE_NUMBER;
      end if;
      l_audit := crypto.get_param('AUDIT_CONTROL') = 'ON' and internal.c_current_user <> crypto.upper_app_schema;      
   END BEFORE STATEMENT;

   AFTER EACH ROW IS
      changes_detected boolean := false;
      l_json varchar2(32767);
   BEGIN
      if l_audit then
         case when updating then
            internal.aud_upd_col (l_json, internal.json_str(:old.job_id),     internal.json_str(:new.job_id),'JOB_ID' , changes_detected, false, 'J');
            internal.aud_upd_col (l_json, internal.json_str(:old.param_id),   internal.json_str(:new.param_id),'PARAM_ID' , changes_detected, false, 'P');
            internal.aud_upd_col (l_json, internal.json_str(:old.param_order),internal.json_str(:new.param_order),'PARAM_ORDER' , changes_detected, false);         
            internal.aud_upd_col (l_json, internal.json_str(:old.dependent_job_id),     internal.json_str(:new.dependent_job_id),'DEPENDENT_JOB_ID' , changes_detected, false, 'J');
            internal.aud_upd_col (l_json, internal.json_str(:old.dependent_param_id),   internal.json_str(:new.dependent_param_id),'DEPENDENT_PARAM_ID' , changes_detected, false, 'P');
            internal.aud_upd_col (l_json, internal.json_str(:old.dependent_param_order),   internal.json_str(:new.dependent_param_order),'DEPENDENT_PARAM_ORDER' , changes_detected, false);         
            internal.aud_upd_col (l_json, internal.json_str(:old.operator),internal.json_str(:new.operator),'OPERATOR' , changes_detected, false);                  
            if changes_detected then
               internal.aud_upd_row('J',:old.job_id,l_table_name, substr(l_json,2));         
            end if;
         when inserting then
            rid_tab(rid_tab.count+1) := :new.rowid;
         when deleting then
            rid_tab(rid_tab.count+1) := :old.rowid;
         end case;
      end if;
   END AFTER EACH ROW;

   AFTER STATEMENT IS
      l_sql varchar2(4000) := q'!select 'J', t.job_id, '"JOB_ID":'||internal.json_str(t.JOB_ID)
||','||'"JOB_NAME":'||(select internal.json_str(j.job_name) from job j where j.job_id = t.job_id)      
||','||'"PARAM_ID":'||internal.json_str(t.PARAM_ID)
||','||'"PARAM_NAME":'||(select internal.json_str(p.name) from parameters p where p.param_id = t.param_id)      
||','||'"PARAM_ORDER":'||internal.json_str(t.PARAM_ORDER)
||','||'"DEPENDENT_JOB_ID":'||internal.json_str(t.DEPENDENT_JOB_ID)
||','||'"DEPENDENT_JOB_NAME":'||(select internal.json_str(j.job_name) from job j where j.job_id = t.DEPENDENT_JOB_ID)      
||','||'"DEPENDENT_PARAM_ID":'||internal.json_str(t.DEPENDENT_PARAM_ID)
||','||'"DEPENDENT_PARAM_NAME":'||(select internal.json_str(p.name) from parameters p where p.param_id = t.DEPENDENT_PARAM_ID)      
||','||'"DEPENDENT_PARAM_ORDER":'||internal.json_str(t.DEPENDENT_PARAM_ORDER)
||','||'"OPERATOR":'||internal.json_str(t.OPERATOR)
     from  job_param_dependencies!';
   BEGIN
      if l_audit then
         case when inserting then
            if rid_tab.count > 0 then
               internal.aud_ins_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab);
            end if;
         when deleting then
            if rid_tab.count > 0 then
               internal.aud_del_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab, p_scn => l_scn);
            end if;
         when updating then
            null;
         end case;
      end if;
   END AFTER STATEMENT;
  
END;
/

CREATE OR REPLACE TRIGGER jp_aud_trig for insert
                                          or 
                                          update
                                          or
                                          delete on job_parameters COMPOUND TRIGGER
                                 rid_tab           internal.rid_tab;
                                 l_scn             number;
                                 l_table_name varchar2(30) := 'JOB_PARAMETERS';
                                 l_audit      boolean;

   BEFORE STATEMENT IS
   BEGIN
      if deleting then
         l_scn := DBMS_FLASHBACK.GET_SYSTEM_CHANGE_NUMBER;
      end if;
      l_audit := crypto.get_param('AUDIT_CONTROL') = 'ON' and internal.c_current_user <> crypto.upper_app_schema;      
   END BEFORE STATEMENT;
   
   AFTER EACH ROW IS
      changes_detected boolean := false;
      l_json varchar2(32767);
   BEGIN
      if l_audit then
         case when updating then
            internal.aud_upd_col (l_json, internal.json_str(:old.job_id),           internal.json_str(:new.job_id),'JOB_ID' , changes_detected, false, 'J');
            internal.aud_upd_col (l_json, internal.json_str(:old.param_id),           internal.json_str(:new.param_id),'PARAM_ID' , changes_detected, false, 'P');
            internal.aud_upd_col (l_json, internal.json_str(:old.param_order),           internal.json_str(:new.param_order),'PARAM_ORDER' , changes_detected, false);         
            internal.aud_upd_col (l_json, internal.json_str(:old.pass_to_job),           internal.json_str(:new.pass_to_job),'PASS_TO_JOB' , changes_detected, false);                  
            if changes_detected then
               internal.aud_upd_row('J',:old.job_id,l_table_name, substr(l_json,2));         
            end if;
         when inserting then
            rid_tab(rid_tab.count+1) := :new.rowid;
         when deleting then
            rid_tab(rid_tab.count+1) := :old.rowid;
         end case;
      end if;
   END AFTER EACH ROW;

   AFTER STATEMENT IS
      l_sql varchar2(4000) := q'!select 'J', t.job_id, '"JOB_ID":'||internal.json_str(t.JOB_ID)
||','||'"JOB_NAME":'||(select internal.json_str(j.job_name) from job j where j.job_id = t.job_id)      
||','||'"PARAM_ID":'||internal.json_str(t.PARAM_ID)
||','||'"PARAM_NAME":'||(select internal.json_str(p.name) from parameters p where p.param_id = t.param_id)      
||','||'"PARAM_ORDER":'||internal.json_str(t.PARAM_ORDER)
||','||'"PASS_TO_JOB":'||internal.json_str(t.PASS_TO_JOB)
                               from  job_parameters!';
   BEGIN
      if l_audit then
         case when inserting then
            if rid_tab.count > 0 then
               internal.aud_ins_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab);
            end if;
         when deleting then
            if rid_tab.count > 0 then
               internal.aud_del_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab, p_scn => l_scn);
            end if;
         when updating then
            null;
         end case;
      end if;
   END AFTER STATEMENT;
  
END;
/

CREATE OR REPLACE TRIGGER jso_aud_trig for insert
                                          or 
                                          update
                                          or
                                          delete on job_status_overrides COMPOUND TRIGGER
                                 rid_tab           internal.rid_tab;
                                 l_scn             number;
                                 l_table_name varchar2(30) := 'JOB_STATUS_OVERRIDES';
                                 l_audit      boolean;

   BEFORE STATEMENT IS
   BEGIN
      if deleting then
         l_scn := DBMS_FLASHBACK.GET_SYSTEM_CHANGE_NUMBER;
      end if;
      l_audit := crypto.get_param('AUDIT_CONTROL') = 'ON' and internal.c_current_user <> crypto.upper_app_schema;      
   END BEFORE STATEMENT;
   
   AFTER EACH ROW IS
      changes_detected boolean := false;
      l_json varchar2(32767);
   BEGIN
      if l_audit then
         case when updating then
            internal.aud_upd_col (l_json, internal.json_str(:old.job_id),           internal.json_str(:new.job_id),'JOB_ID' , changes_detected, false, 'J');
            internal.aud_upd_col (l_json, internal.json_str(:old.search_string), internal.json_str(:new.search_string),'SEARCH_STRING', changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.match), internal.json_str(:new.match),'MATCH', changes_detected, false);                                    
            internal.aud_upd_col (l_json, internal.json_str(:old.mark_as), internal.json_str(:new.mark_as),'MARK_AS', changes_detected, false);         
            internal.aud_upd_col (l_json, internal.json_str(:old.proc_order), internal.json_str(:new.proc_order),'PROC_ORDER', changes_detected, false);                  
            if changes_detected then
               internal.aud_upd_row('J',:old.job_id,l_table_name, substr(l_json,2));
            end if;
         when inserting then
            rid_tab(rid_tab.count+1) := :new.rowid;
         when deleting then
            rid_tab(rid_tab.count+1) := :old.rowid;
         end case;
      end if;
   END AFTER EACH ROW;

   AFTER STATEMENT IS
      l_sql  varchar2(4000) := q'!select 'J', t.job_id, '"JOB_ID":'||internal.json_str(t.JOB_ID)
||','||'"JOB_NAME":'||(select internal.json_str(j.job_name) from job j where j.job_id = t.job_id)            
||','||'"SEARCH_STRING":'||internal.json_str(t.SEARCH_STRING)
||','||'"MATCH":'||internal.json_str(t.MATCH)
||','||'"MARK_AS":'||internal.json_str(t.MARK_AS)
||','||'"PROC_ORDER":'||internal.json_str(t.PROC_ORDER)
from   job_status_overrides!';
   BEGIN
      if l_audit then
         case when inserting then
            if rid_tab.count > 0 then
               internal.aud_ins_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab);
            end if;
         when deleting then
            if rid_tab.count > 0 then
               internal.aud_del_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab, p_scn => l_scn);
            end if;
         when updating then
            null;
         end case;
      end if;
   END AFTER STATEMENT;
  
END;
/

CREATE OR REPLACE TRIGGER j_aud_trig for insert
                                          or 
                                          update
                                          or
                                          delete on job COMPOUND TRIGGER
                                 rid_tab           internal.rid_tab;
                                 l_scn             number;
                                 l_table_name varchar2(30) := 'JOB';
                                 l_audit      boolean;

   BEFORE STATEMENT IS
   BEGIN
      if deleting then
         l_scn := DBMS_FLASHBACK.GET_SYSTEM_CHANGE_NUMBER;
      end if;
      l_audit := crypto.get_param('AUDIT_CONTROL') = 'ON' and internal.c_current_user <> crypto.upper_app_schema;      
   END BEFORE STATEMENT;
   
   AFTER EACH ROW IS
      changes_detected boolean := false;
      l_json varchar2(32767);
   BEGIN
      if l_audit then
         case when updating then
            internal.aud_upd_col (l_json, internal.json_str(:old.job_id), internal.json_str(:new.job_id),'JOB_ID' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.job_online), internal.json_str(:new.job_online),'JOB_ONLINE' , changes_detected, false);         
            internal.aud_upd_col (l_json, internal.json_str(:old.job_name), internal.json_str(:new.job_name),'JOB_NAME' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.priority), internal.json_str(:new.priority),'PRIORITY' , changes_detected, false);         
            internal.aud_upd_col (l_json, internal.json_str(:old.max_complete_exit_code), internal.json_str(:new.max_complete_exit_code),'MAX_COMPLETE_EXIT_CODE' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.runtime_warning), internal.json_str(:new.runtime_warning),'RUNTIME_WARNING' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.resub_on_dbstart), internal.json_str(:new.resub_on_dbstart),'RESUB_ON_DBSTART' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.overdue_behaviour), internal.json_str(:new.overdue_behaviour),'OVERDUE_BEHAVIOUR' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.job_description), internal.json_str(:new.job_description),'JOB_DESCRIPTION' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.doc_link), internal.json_str(:new.doc_link),'DOC_LINK' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.msg_text), internal.json_str(:new.msg_text),'MSG_TEXT' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.source_type), internal.json_str(:new.source_type),'SOURCE_TYPE' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.ssh_conn_string), internal.json_str(:new.ssh_conn_string),'SSH_CONN_STRING' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.source_dir), internal.json_str(:new.source_dir),'SOURCE_DIR' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.source_name), internal.json_str(:new.source_name),'SOURCE_NAME' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.username), internal.json_str(:new.username),'USERNAME' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.password), internal.json_str(:new.password),'PASSWORD' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.db_conn_string), internal.json_str(:new.db_conn_string),'DB_CONN_STRING' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.connect_as), internal.json_str(:new.connect_as),'CONNECT_AS' , changes_detected, false);         
            internal.aud_upd_col (l_json, internal.json_str(:old.schedule_type), internal.json_str(:new.schedule_type),'SCHEDULE_TYPE' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.parent_job_id), internal.json_str(:new.parent_job_id),'PARENT_JOB_ID' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.next_sub_dttm), internal.json_str(:new.next_sub_dttm),'NEXT_SUB_DTTM' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.exe_window_start_dttm), internal.json_str(:new.exe_window_start_dttm),'EXE_WINDOW_START_DTTM' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.exe_window_end_dttm), internal.json_str(:new.exe_window_end_dttm),'EXE_WINDOW_END_DTTM' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.job_frequency), internal.json_str(:new.job_frequency),'JOB_FREQUENCY' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.job_frequency_sql), internal.json_str(:new.job_frequency_sql),'JOB_FREQUENCY_SQL' , changes_detected, false);         
            internal.aud_upd_col (l_json, internal.json_str(:old.job_cron_str), internal.json_str(:new.job_cron_str),'JOB_CRON_STR' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.email), internal.json_str(:new.email),'EMAIL' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.email_log), internal.json_str(:new.email_log),'EMAIL_LOG' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.completion_message), internal.json_str(:new.completion_message),'COMPLETION_MESSAGE' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.email_on_failure), internal.json_str(:new.email_on_failure),'EMAIL_ON_FAILURE' , changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.email_on_failure_cc_ops), internal.json_str(:new.email_on_failure_cc_ops),'EMAIL_ON_FAILURE_CC_OPS' , changes_detected, false);
            if changes_detected then
               internal.aud_upd_row('J',:old.job_id,l_table_name, substr(l_json,2));
            end if;
         when inserting then
            rid_tab(rid_tab.count+1) := :new.rowid;
         when deleting then
            rid_tab(rid_tab.count+1) := :old.rowid;
         end case;
      end if;
   END AFTER EACH ROW;

   AFTER STATEMENT IS
      l_sql  varchar2(4000) := q'!select 'J', t.job_id, '"JOB_ID":'||internal.json_str(t.JOB_ID)
||','||'"JOB_ONLINE":'||internal.json_str(t.JOB_ONLINE)      
||','||'"JOB_NAME":'||internal.json_str(t.JOB_NAME)
||','||'"PRIORITY":'||internal.json_str(t.PRIORITY)
||','||'"MAX_COMPLETE_EXIT_CODE":'||internal.json_str(t.MAX_COMPLETE_EXIT_CODE)
||','||'"RUNTIME_WARNING":'||internal.json_str(t.RUNTIME_WARNING)
||','||'"RESUB_ON_DBSTART":'||internal.json_str(t.RESUB_ON_DBSTART)
||','||'"OVERDUE_BEHAVIOUR":'||internal.json_str(t.OVERDUE_BEHAVIOUR)
||','||'"JOB_DESCRIPTION":'||internal.json_str(t.JOB_DESCRIPTION)
||','||'"DOC_LINK":'||internal.json_str(t.DOC_LINK)
||','||'"MSG_TEXT":'||internal.json_str(t.MSG_TEXT)
||','||'"SOURCE_TYPE":'||internal.json_str(t.SOURCE_TYPE)
||','||'"SSH_CONN_STRING":'||internal.json_str(t.SSH_CONN_STRING)
||','||'"SOURCE_DIR":'||internal.json_str(t.SOURCE_DIR)
||','||'"SOURCE_NAME":'||internal.json_str(t.SOURCE_NAME)
||','||'"USERNAME":'||internal.json_str(t.USERNAME)
||','||'"PASSWORD":'||internal.json_str(t.PASSWORD)
||','||'"DB_CONN_STRING":'||internal.json_str(t.DB_CONN_STRING)
||','||'"CONNECT_AS":'||internal.json_str(t.CONNECT_AS)
||','||'"SCHEDULE_TYPE":'||internal.json_str(t.SCHEDULE_TYPE)
||','||'"PARENT_JOB_ID":'||internal.json_str(t.PARENT_JOB_ID)
||','||'"NEXT_SUB_DTTM":'||internal.json_str(t.NEXT_SUB_DTTM)
||','||'"EXE_WINDOW_START_DTTM":'||internal.json_str(t.EXE_WINDOW_START_DTTM)
||','||'"EXE_WINDOW_END_DTTM":'||internal.json_str(t.EXE_WINDOW_END_DTTM)
||','||'"JOB_FREQUENCY":'||internal.json_str(t.JOB_FREQUENCY)
||','||'"JOB_FREQUENCY_SQL":'||internal.json_str(t.JOB_FREQUENCY_SQL)
||','||'"JOB_CRON_STR":'||internal.json_str(t.JOB_CRON_STR)
||','||'"EMAIL":'||internal.json_str(t.EMAIL)
||','||'"EMAIL_LOG":'||internal.json_str(t.EMAIL_LOG)
||','||'"COMPLETION_MESSAGE":'||internal.json_str(t.COMPLETION_MESSAGE)
||','||'"EMAIL_ON_FAILURE":'||internal.json_str(t.EMAIL_ON_FAILURE)
||','||'"EMAIL_ON_FAILURE_CC_OPS":'||internal.json_str(t.EMAIL_ON_FAILURE_CC_OPS)
from   job!';
   BEGIN
      if l_audit then
         case when inserting then
            if rid_tab.count > 0 then
               internal.aud_ins_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab);
            end if;
         when deleting then
            if rid_tab.count > 0 then
               internal.aud_del_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab, p_scn => l_scn);
            end if;
         when updating then
            null;
         end case;
      end if;
   END AFTER STATEMENT;
  
END;
/

CREATE OR REPLACE TRIGGER cat_aud_trig for insert
                                          or 
                                          update
                                          or
                                          delete on console_auto_text COMPOUND TRIGGER
                                 rid_tab           internal.rid_tab;
                                 l_scn             number;
                                 l_table_name varchar2(30) := 'CONSOLE_AUTO_TEXT';
                                 l_audit      boolean;

   BEFORE STATEMENT IS
   BEGIN
      if deleting then
         l_scn := DBMS_FLASHBACK.GET_SYSTEM_CHANGE_NUMBER;
      end if;
      l_audit := true;
   END BEFORE STATEMENT;

   AFTER EACH ROW IS
      changes_detected boolean := false;
      l_json varchar2(32767);
   BEGIN
      if l_audit then
         case when updating then
            internal.aud_upd_col (l_json, internal.json_str(:old.auto_text),internal.json_str(:new.auto_text),'AUTO_TEXT',changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.username), internal.json_str(:new.username),'USERNAME', changes_detected, false);
            internal.aud_upd_col (l_json, internal.json_str(:old.timestamp), internal.json_str(:new.timestamp),'TIMESTAMP', changes_detected, false);         
            if changes_detected then
               internal.aud_upd_row('P',-999,l_table_name, substr(l_json,2));
            end if;
         when inserting then
            rid_tab(rid_tab.count+1) := :new.rowid;
         when deleting then
            rid_tab(rid_tab.count+1) := :old.rowid;
         end case;
      end if;
   END AFTER EACH ROW;

   AFTER STATEMENT IS
      l_sql  varchar2(4000) := q'!select 'S', -999, '"AUTO_TEXT":'||internal.json_str(t.AUTO_TEXT)
||','||'"USERNAME":'||internal.json_str(t.USERNAME)
||','||'"TIMESTAMP":'||internal.json_str(t.TIMESTAMP)
from   console_auto_text!';
   BEGIN
      if l_audit then
         case when inserting then
            if rid_tab.count > 0 then
               internal.aud_ins_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab);
            end if;
         when deleting then
            if rid_tab.count > 0 then
               internal.aud_del_rows(p_table_name=>l_table_name, p_sql=> l_sql, p_rids => rid_tab, p_scn => l_scn);
            end if;
         when updating then
            null;
         end case;
      end if;
   END AFTER STATEMENT;
END;
/

